/* Copyright (c) 2004, 2011, Oracle and/or its affiliates. 
All rights reserved. */

/*

   NAME
     crsapp.c - Demo application using CRS api

   DESCRIPTION
     This program demonstrates the usage of CRS api (defined in clscrsx.h) to use 
     the High Availability infrastructure provided by CRS.

     This program does the following things using the CLSCRS APIs:
     1. Create resource types.
     2. Create resources of type local_resource, application, and the 
        resource types created in step 1.
     3. Create a server pool.
     4. Register resources.
     5. Register resource types.
     6. Register server pool.
     7. Start resources using 2 methods:
        a. By specifying resource name explicitly.
        b. By specifying a filter.
     8. Stat resource types by using both type names and filters.
     9. Stat resources by using both resource names and filters.
     10. Stat server pools
     11. Stat servers
     12. Execute a check action on resources.
     13. Modify a resource.
     14. Relocate a resource to a different server.
     15. Stop resources using 2 methods:
         a. By specifying resource name explicitly.
         b. By specifying a filter.
     16. Stat multiple CRS entities with a single call.    
     17. Unregister resources.
     18. Unregister resource types.
     19. Unregister server pools.
        

   INTERNAL FUNCTION(S)

   NOTES
      The action script needs to be at ORACLE_HOME\crs\demo.
      For Windows, the action script file is action_scr.bat.
      For Linux, the action script file is action_scr.scr.

   MODIFIED   (MM/DD/YY)
   smatpadi    07/08/11 - Bug/11821124: ICC compiler crash
   gdbhat      01/21/11 - Changed findtype to get_type
   vmanivel    12/21/10 - WhatIf stat3
   gdbhat      01/10/10 - Stat 3 code
   gdbhat      02/22/10 - NT Fix
   gdbhat      10/19/09 - Update app with new APIs
   rdasari     02/06/06 - pass traceback in init 
   rajayar     09/08/05 - bug fixes 4448442 
   rdasari     08/03/04 - rdasari_api_app
   rdasari     07/22/04 - Creation

*/

# include <stdio.h>
# include <stdlib.h>

#ifndef CLSCRSX_ORACLE
# include <clscrsx.h>
#endif

/*---------------------------------------------------------------------------
                     PRIVATE TYPES AND CONSTANTS
  ---------------------------------------------------------------------------*/

#define MAXPATHNAMELEN 80
#define MAXNODENAMELEN 20

#define RESOURCE_1 (const oratext *)"clscrs.resource1.demo"
#define RESOURCE_2 (const oratext *)"clscrs.resource2.demo"
#define RESOURCE_3 (const oratext *)"clscrs.resource3.demo"
#define RESOURCE_4 (const oratext *)"clscrs.resource4.demo"
#define RESOURCE_5 (const oratext *)"clscrs.resource5.demo"

#define RESOURCE_TYPE_1 (const oratext *)"clscrs.resourcetype1.demo"
#define RESOURCE_TYPE_2 (const oratext *)"clscrs.resourcetype2.demo"

#define SERVER_POOL_1 (const oratext *)"clscrs.serverpool1.demo"
#define SEPARATOR     "********** ********** ********** ********** ********** ********** ********** \n" 

#ifdef WIN32COMMON
# define FILE_EXTENSION (const oratext *)"bat"
#else
# define FILE_EXTENSION (const oratext *)"scr"
#endif

/*---------------------------------------------------------------------------
                     HELPER FUNCTIONS START
  ---------------------------------------------------------------------------*/

/* This function prints out the contents of a reslist

  APIs Used:
    clscrs_reslist_count
    clscrs_reslist_first
    clscrs_reslist_next
    clscrs_res_get_name
    clscrs_splist_count
    clscrs_splist_first
    clscrs_splist_next
    clscrs_sp_get
    clscrs_type_get_attr
*/
void print_entitylist(clscrs_reslist *entity_list, boolean is_type)
{
  clscrs_res    *curr_entity;
  ub4            count;
  oratext       *curr_entityname = NULL, *mesg = NULL;
  clscrs_splist *attr_list;
  clscrs_sp     *attr;
  oratext        default_value[4096], *attrname, *attrvalue;
  CLSCRS_STAT    status;
  boolean        reg_status;
  uword          flags = 0;
  clscrs_dtype   data_type;

  clscrs_reslist_count(entity_list, &count);
  printf("Number of entities = %d\n", count);

  if (count == 0) 
  {
    return;
  }

  printf("Printing entity list:\n");
  for (clscrs_reslist_first(entity_list, &curr_entity); curr_entity ;
       clscrs_reslist_next(entity_list, &curr_entity)) 
  {
    clscrs_res_get_name(curr_entity, &curr_entityname);
    printf("Entity name = %s\n", curr_entityname);

    clscrs_res_get_op_status(curr_entity, &status, &mesg);
    printf("Operation status = %d\n", status);

    clscrs_res_get_registered(curr_entity, &reg_status);
    printf("Registration status = %d\n", reg_status);

    clscrs_res_get_attr_list(curr_entity, &attr_list);
    clscrs_splist_count(attr_list, &count);
    printf("%s : %d %s Attribute Count = %d\n", curr_entityname, status, 
              mesg, count);

    if (count == 0) 
    {
      return;
    }

    printf("Attrlist for entity %s:\n", curr_entityname);
    for (clscrs_splist_first(attr_list, &attr); attr ;
         clscrs_splist_next(attr_list, &attr)) 
    {
      clscrs_sp_get(attr, &attrname, &attrvalue);

      if (!is_type)
      {
        printf("%s = %s \n", attrname, attrvalue);
      }
      else 
      {
        memset(default_value, 0, 4096);
        clscrs_type_get_attr(curr_entity, attrname, &data_type, default_value,
                             &flags);

        printf("Attribute name = %s\n", attrname);
        printf("Attribute data type = %d\n", data_type);
        printf("Attribute default value = %s\n", default_value);
        printf("Attribute flags = %d\n", flags);
      }
    }
    printf("\n");
  }
}

/* print function for stat 3 */
void print_entitylist_stat3(clscrs_reslist *entity_list)
{
  clscrs_res    *curr_entity;
  ub4            count;
  oratext       *curr_entityname = NULL, *mesg = NULL, *tmpname = NULL;
  clscrs_splist *attr_list;
  clscrs_sp     *attr;
  oratext        default_value[4096], *attrname, *attrvalue;
  CLSCRS_STAT    status;
  boolean        reg_status;
  uword          flags = 0;
  clscrs_dtype   data_type;
  clscrs_reslist *pResList;
  clscrs_entity_type entity_type;

  clscrs_reslist_count(entity_list, &count);
  printf("Number of CRS entities = %d\n", count);

  if (count == 0) 
  {
    return;
  }

  printf("Printing stat3 output entity list:\n");
  printf("------------------------------------\n");
  for (clscrs_reslist_first(entity_list, &curr_entity); curr_entity ;
       clscrs_reslist_next(entity_list, &curr_entity)) 
  {
    clscrs_res_get_reslist(curr_entity, &pResList);
        
    clscrs_res_get_name(curr_entity, &curr_entityname);
    printf("Entity name = %s\n", curr_entityname);
    printf("------------------------------------\n");
   
    if (pResList == NULL)
    {
      printf("Number of entities for %s = 0\n", curr_entityname);
      printf("------------------------------------\n");
      continue;
    }

    clscrs_reslist_count(pResList, &count);
    printf("Number of entities for %s = %d\n", curr_entityname, count);

    if (count == 0) 
    {
      continue;
    }

    clscrs_crsentity_get_type(curr_entity, &entity_type);
    if (entity_type == clscrs_entity_restype)
    {
      print_entitylist(pResList, TRUE);
    }
    else
    {
      print_entitylist(pResList, FALSE);
    }
                                    
    printf("------------------------------------\n");
  }
}

/* Interactive APIs in CLSCRS provide a callback mechanism that the clients
   can use to process error messages, warning messages and progress
   messages sent by the CRSD.
*/

void client_callback(void *arg, const oratext *pId, const oratext *pMsg,
                     clscrs_msgtype msgType)
{
  if (pMsg != NULL)
  {
     printf("%s\n", pMsg);
  }
}

/* This function prints out the status of the operations performed by 
   the Oracle Clusterware on the CRS entities.
*/
void print_opstatus(clscrs_reslist *op_status)
{
  clscrs_res  *res = NULL;
  oratext     *resname, *mesg;
  CLSCRS_STAT  status;

  printf("\nPrinting op_status list:\n");

  for (clscrs_reslist_first(op_status, &res); res ; 
       clscrs_reslist_next(op_status, &res))
  {
    clscrs_res_get_name(res, &resname);
    clscrs_res_get_op_status(res, &status, &mesg);
    printf("Status of operation on %s : %d\n", resname, status);
  }

  printf("\n");
  return;
}

/* Function to print clscrs API trace msgs */

void print(void *ctx, oratext *fmt, ...)
{
  printf(fmt);
  printf("\n");
}

/*---------------------------------------------------------------------------
                     HELPER FUNCTIONS END
  ---------------------------------------------------------------------------*/

/*
 This function creates the CRS entities to be used in this demo application.

 APIs Used:
  clscrs_res_create
  clscrs_type_create
  clscrs_res_set_attr
  clscrs_type_set_attr
  clscrs_reslist_append
  clscrs_getnodename
  clscrs_get_crshome
*/

void create_crs_entities(clscrs_ctx     *ctx,
                         clscrs_reslist *res_list,
                         clscrs_reslist *res_type_list,
                         clscrs_reslist *server_pool_list)
{
  clscrs_res *res1         = NULL, *res2 = NULL, *res3 = NULL, *res4 = NULL;
  clscrs_res *res5         = NULL;
  clscrs_res *res_type1    = NULL, *res_type2 = NULL;
  clscrs_res *server_pool1 = NULL;
  oratext    *script_name  = (oratext *)"action_scr";
  size_t      size         = MAXPATHNAMELEN;
  oratext     crshome[MAXPATHNAMELEN];
  oratext     action_script[MAXPATHNAMELEN];
  oratext     script_path[MAXPATHNAMELEN];
  oratext     nodename[MAXNODENAMELEN];
  CLSCRS_STAT status;

  /* Get the crshome */
  if (CLSCRS_STAT_SUCCESS != 
      (status = clscrs_get_crshome(ctx, crshome, &size))) 
  {
    printf("clscrs_get_crshome failed with status %d\n", status);
    exit(1);
  }
  
  /* Get the complete pathname for the action script */
  clsp_addToPath(script_path, MAXPATHNAMELEN, crshome, 
                 (oratext *)"crs", (oratext *)"demo", (oratext *)0);
  clsp_getFile(action_script, MAXPATHNAMELEN, script_path, script_name, 
               FILE_EXTENSION);

  /* Get the local nodename */
  clscrs_getnodename(ctx, nodename);

  /* Create the Resource Types */
  clscrs_type_create(ctx, RESOURCE_TYPE_1, &res_type1);
  clscrs_type_set_attr(res_type1, (oratext *)"AUTO_START",clscrs_dtype_STRING,
                       "always", 0);
  clscrs_type_set_attr(res_type1, (oratext *)"BASE_TYPE",clscrs_dtype_STRING,
                       "local_resource", 0);
 
  clscrs_type_create(ctx, RESOURCE_TYPE_2, &res_type2);
  clscrs_type_set_attr(res_type2, (oratext *)"AUTO_START",clscrs_dtype_STRING,
                       "never", 0);
  clscrs_type_set_attr(res_type2, (oratext *)"BASE_TYPE",clscrs_dtype_STRING,
                       "cluster_resource", 0);

  /* Add the resource types to a resource type list */
  clscrs_reslist_append(res_type_list, res_type1);
  clscrs_reslist_append(res_type_list, res_type2);

  /* Create the Resources */
  clscrs_res_create(ctx, RESOURCE_1, &res1);
  clscrs_res_set_attr(res1, clscrs_NAME, RESOURCE_1);
  clscrs_res_set_attr(res1, clscrs_TYPE, (oratext *)"application");
  clscrs_res_set_attr(res1, clscrs_ACTION_SCRIPT, action_script);
  clscrs_res_set_attr(res1, clscrs_SCRIPT_TIMEOUT, (oratext *)"50");
  clscrs_res_set_attr(res1, clscrs_AUTO_START, (oratext *)"always");
  clscrs_res_set_attr(res1, clscrs_DESCRIPTION, RESOURCE_1);
  clscrs_res_set_attr(res1, clscrs_PLACEMENT, (oratext *)"favored");
  clscrs_res_set_attr(res1, clscrs_FAILOVER_DELAY, (oratext *)"10");
  clscrs_res_set_attr(res1, clscrs_FAILURE_THRESHOLD, (oratext *)"10");
  clscrs_res_set_attr(res1, clscrs_FAILURE_INTERVAL, (oratext *)"10");
  clscrs_res_set_attr(res1, clscrs_ACTIVE_PLACEMENT, (oratext *)"1");
  clscrs_res_set_attr(res1, clscrs_REQUIRED_RESOURCES, (oratext *)" ");
  clscrs_res_set_attr(res1, clscrs_OPTIONAL_RESOURCES, (oratext *)" ");
  clscrs_res_set_attr(res1, clscrs_HOSTING_MEMBERS, nodename);
  clscrs_res_set_attr(res1, clscrs_CHECK_INTERVAL, (oratext *)"50");
  clscrs_res_set_attr(res1, clscrs_RESTART_ATTEMPTS, (oratext *)"1");
  
  clscrs_res_create(ctx, RESOURCE_2, &res2);
  clscrs_res_set_attr(res2, clscrs_NAME, RESOURCE_2);
  clscrs_res_set_attr(res2, clscrs_TYPE, RESOURCE_TYPE_1);
  clscrs_res_set_attr(res2, clscrs_ACTION_SCRIPT, action_script);
  clscrs_res_set_attr(res2, clscrs_SCRIPT_TIMEOUT, (oratext *)"50");
  clscrs_res_set_attr(res2, clscrs_AUTO_START, (oratext *)"never");
  clscrs_res_set_attr(res2, clscrs_DESCRIPTION, RESOURCE_2);
  clscrs_res_set_attr(res2, clscrs_CHECK_INTERVAL, (oratext *)"50");
  clscrs_res_set_attr(res2, clscrs_RESTART_ATTEMPTS, (oratext *)"1");

  clscrs_res_create(ctx, RESOURCE_3, &res3);
  clscrs_res_set_attr(res3, clscrs_NAME, RESOURCE_3);
  clscrs_res_set_attr(res3, clscrs_TYPE, RESOURCE_TYPE_2);
  clscrs_res_set_attr(res3, clscrs_ACTION_SCRIPT, action_script);
  clscrs_res_set_attr(res3, clscrs_SCRIPT_TIMEOUT, (oratext *)"50");
  clscrs_res_set_attr(res3, clscrs_AUTO_START, (oratext *)"never");
  clscrs_res_set_attr(res3, clscrs_DESCRIPTION, RESOURCE_3);
  clscrs_res_set_attr(res3, clscrs_PLACEMENT, (oratext *)"balanced");
  clscrs_res_set_attr(res3, clscrs_FAILOVER_DELAY, (oratext *)"10");
  clscrs_res_set_attr(res3, clscrs_FAILURE_THRESHOLD, (oratext *)"10");
  clscrs_res_set_attr(res3, clscrs_ACTIVE_PLACEMENT, (oratext *)"1");
  clscrs_res_set_attr(res3, clscrs_HOSTING_MEMBERS, nodename);
  clscrs_res_set_attr(res3, clscrs_CHECK_INTERVAL, (oratext *)"50");
  clscrs_res_set_attr(res3, clscrs_RESTART_ATTEMPTS, (oratext *)"1");
  clscrs_res_set_attr(res3, clscrs_CARDINALITY, (oratext *)"2");

  clscrs_res_create(ctx, RESOURCE_4, &res4);
  clscrs_res_set_attr(res4, clscrs_NAME, RESOURCE_4);
  clscrs_res_set_attr(res4, clscrs_TYPE, RESOURCE_TYPE_2);
  clscrs_res_set_attr(res4, clscrs_ACTION_SCRIPT, action_script);
  clscrs_res_set_attr(res4, clscrs_SCRIPT_TIMEOUT, (oratext *)"50");
  clscrs_res_set_attr(res4, clscrs_AUTO_START, (oratext *)"never");
  clscrs_res_set_attr(res4, clscrs_DESCRIPTION, RESOURCE_4);
  clscrs_res_set_attr(res4, clscrs_PLACEMENT, (oratext *)"favored");
  clscrs_res_set_attr(res4, clscrs_FAILOVER_DELAY, (oratext *)"10");
  clscrs_res_set_attr(res4, clscrs_FAILURE_THRESHOLD, (oratext *)"10");
  clscrs_res_set_attr(res4, clscrs_HOSTING_MEMBERS, nodename);
  clscrs_res_set_attr(res4, clscrs_CHECK_INTERVAL, (oratext *)"50");
  clscrs_res_set_attr(res4, clscrs_RESTART_ATTEMPTS, (oratext *)"1");

  clscrs_res_create(ctx, RESOURCE_5, &res5);
  clscrs_res_set_attr(res5, clscrs_NAME, RESOURCE_5);
  clscrs_res_set_attr(res5, clscrs_TYPE, (oratext *)"local_resource");
  clscrs_res_set_attr(res5, clscrs_ACTION_SCRIPT, action_script);
  clscrs_res_set_attr(res2, clscrs_SCRIPT_TIMEOUT, (oratext *)"50");
  clscrs_res_set_attr(res5, clscrs_AUTO_START, (oratext *)"never");
  clscrs_res_set_attr(res5, clscrs_DESCRIPTION, RESOURCE_5);
  clscrs_res_set_attr(res5, clscrs_RESTART_ATTEMPTS, (oratext *)"1");

  /* Add the resources to a resource list */
  clscrs_reslist_append(res_list, res1);
  clscrs_reslist_append(res_list, res2);
  clscrs_reslist_append(res_list, res3);
  clscrs_reslist_append(res_list, res4);
  clscrs_reslist_append(res_list, res5);

  /* Create the Server Pool */

  clscrs_serverpool_create(ctx, SERVER_POOL_1, &server_pool1);
  clscrs_res_set_attr(server_pool1, clscrs_NAME, SERVER_POOL_1);
        
  /* Add the server pool to a serverpool list */
  clscrs_reslist_append(server_pool_list, server_pool1);

}

/*
 This function registers the demo resource types with Oracle Clusterware.

 APIs Used:
  clscrs_register_type
  clscrs_reslist_create
  clscrs_reslist_destroy
*/

void register_resource_types(clscrs_ctx     *ctx, 
                             clscrs_reslist *restype_list)
{
  clscrs_reslist *op_status = NULL;
  CLSCRS_STAT     status;

  printf("Registering resource types\n");

  clscrs_reslist_create(ctx, &op_status);

  status = clscrs_register_type(restype_list, 0,
                                (clscrs_msgf2)client_callback,
                                NULL, op_status);

  printf("\nRegister resource type status = %d\n", status);
  
  print_opstatus(op_status);
  
  clscrs_reslist_destroy(&op_status);
  
}

/*
 This function registers the demo resources with Oracle Clusterware.

 APIs Used:
  clscrs_register_resource2
  clscrs_reslist_create
  clscrs_reslist_destroy
*/

void register_resources(clscrs_ctx     *ctx, 
                               clscrs_reslist *reslist)
{
  clscrs_reslist *op_status = NULL;
  CLSCRS_STAT     status;

  printf("Registering resources\n");

  clscrs_reslist_create(ctx, &op_status);

  status = clscrs_register_resource2(reslist, 0,
                                     (clscrs_msgf2)client_callback,
                                     NULL, op_status);

  printf("\nRegister resource status = %d\n", status);
  
  print_opstatus(op_status);
  
  clscrs_reslist_destroy(&op_status);
  
}

/*
 This function registers the demo server pools with Oracle Clusterware.

 APIs Used:
  clscrs_register_serverpool
  clscrs_reslist_create
  clscrs_reslist_destroy
*/

void register_server_pools(clscrs_ctx     *ctx, 
                                  clscrs_reslist *splist)
{
  clscrs_reslist *op_status = NULL;
  CLSCRS_STAT     status;

  printf("Registering server pool\n");

  clscrs_reslist_create(ctx, &op_status);

  status = clscrs_register_serverpool(splist, 0,
                                     (clscrs_msgf2)client_callback,
                                     NULL, op_status);

  printf("\nRegister server pool status = %d\n", status);
  
  print_opstatus(op_status);
  
  clscrs_reslist_destroy(&op_status);
  
}

/*
  This function is used to start the resources registered with Oracle 
  Clusterware. It employs both filters as well as regular resource names.

  APIs Used:
    clscrs_start_resource2
    clscrs_compfilter_create
    clscrs_exprfilter_create
    clscrs_filter_print
    clscrs_filter_destroy
    clscrs_env_create
    clscrs_entity_id_create
*/
void start_resources(clscrs_ctx *ctx)
{
  oratext          *res_id1     = NULL;
  clscrs_reslist   *op_status   = NULL;
  clscrs_splist    *resid_list  = NULL;
  CLSCRS_STAT       status;
  clscrs_filter    *filter1     = NULL, *filter2 = NULL;
  clscrs_filter    *expr_filter = NULL;
  clscrs_comparator comp_ne     = clscrs_comparator_ne;
  clscrs_comparator comp_eq     = clscrs_comparator_eq;
  clscrs_operator   and_filter  = clscrs_operator_and;
  clscrs_env        env;

  printf("\nStarting resources\n");

  clscrs_env_create(ctx, &env);

  /* Start a single resource */
  clscrs_entity_id_create(ctx, RESOURCE_1, clscrs_entity_res, &res_id1);

  clscrs_splist_create(ctx, &resid_list);
  clscrs_splist_append(resid_list, res_id1, NULL);

  clscrs_reslist_create(ctx, &op_status);

  status = clscrs_start_resource2(resid_list, NULL, env, 
                                  (clscrs_msgf2)client_callback, 
                                  NULL, 0,
                                  op_status);

  print_opstatus(op_status);

  clscrs_reslist_destroy(&op_status);
  clscrs_splist_destroy(&resid_list);
  clscrs_entity_id_destroy(ctx, &res_id1);

  /* Start resources matchihng a particular filter */
  /* Create a simple filter TYPE == clscrs.resourcetype2.demo */
  clscrs_compfilter_create(ctx, clscrs_TYPE, 
                           comp_eq, RESOURCE_TYPE_2, &filter1);
  clscrs_filter_print(filter1);
  
  /* Create another simple filter PLACEMENT != balanced */
  clscrs_compfilter_create(ctx, clscrs_PLACEMENT, 
                              comp_ne, "balanced", &filter2);
  clscrs_filter_print(filter2);
  
  /* Create an expression filter:
     TYPE == clscrs.resourcetype2.demo && PLACEMENT != balanced 
  */
  clscrs_exprfilter_create(filter1, and_filter, filter2, &expr_filter);
  clscrs_filter_print(expr_filter);

  /* Start resources matching the expression filter specified in expr_filter */
  clscrs_reslist_create(ctx, &op_status);
  status = clscrs_start_resource2(expr_filter, NULL, env, 
                                  (clscrs_msgf2)client_callback, 
                                  NULL, 0, op_status);
  print_opstatus(op_status);

  clscrs_filter_destroy(&filter1);
  clscrs_filter_destroy(&filter2);
  clscrs_filter_destroy(&expr_filter);
  clscrs_reslist_destroy(&op_status);

  /* Start a single instance of a resource having multiple cardinality */
  /* Create a simple filter NAME == clscrs.resource3.demo */
  clscrs_compfilter_create(ctx, clscrs_NAME, 
                           comp_eq, RESOURCE_3, &filter1);
  clscrs_filter_print(filter1);
  
  /* Create another simple filter CARDINALITY_ID == 1 */
  clscrs_compfilter_create(ctx, clscrs_CARDINALITY_ID, 
                              comp_eq, "1", &filter2);
  clscrs_filter_print(filter2);
  
  /* Create an expression filter:
     NAME == clscrs.resource3.demo && CARDINALITY_ID == 1 
  */
  clscrs_exprfilter_create(filter1, and_filter, filter2, &expr_filter);
  clscrs_filter_print(expr_filter);

  /* Start a single instance of a resource */
  clscrs_reslist_create(ctx, &op_status);
  status = clscrs_start_resource2(expr_filter, NULL, env, 
                                  (clscrs_msgf2)client_callback, 
                                  NULL, 0, op_status);
  print_opstatus(op_status);

  clscrs_filter_destroy(&filter1);
  clscrs_filter_destroy(&filter2);
  clscrs_filter_destroy(&expr_filter);
  clscrs_reslist_destroy(&op_status);
  clscrs_env_delete(env);
}

/*
  This function is used to get status of resource types. It employs both 
  filters as well as regular resource type names.

  APIs Used:
    clscrs_stat2
    clscrs_compfilter_create
    clscrs_exprfilter_create
    clscrs_filter_print
*/
void stat_resource_types(clscrs_ctx *ctx)
{
  clscrs_reslist   *out_entitylist;
  clscrs_splist    *attrlist, *rqlist;
  CLSCRS_STAT       status;
  clscrs_filter    *filter1 = NULL;
  clscrs_comparator comp_eq = clscrs_comparator_eq;
  oratext          *res_type_id = NULL;

  printf("\nQuerying resource types\n");

  /* Stat a single resource type asking for all attributes*/
  clscrs_reslist_create(ctx, &out_entitylist);
  clscrs_entity_id_create(ctx, RESOURCE_TYPE_1, clscrs_entity_restype, &res_type_id);
  clscrs_splist_create(ctx, &rqlist);
  clscrs_splist_append(rqlist, res_type_id, NULL);
  status = clscrs_stat2(rqlist, clscrs_entity_restype, NULL, NULL, 
                        NULL, 0,
                        out_entitylist);
  print_entitylist(out_entitylist, TRUE);

  clscrs_splist_destroy(&rqlist);
  clscrs_reslist_destroy(&out_entitylist);
  clscrs_entity_id_destroy(ctx, &res_type_id);

  /* Stat all resource types asking for specific attributes*/
  clscrs_reslist_create(ctx, &out_entitylist);
  clscrs_splist_create(ctx, &attrlist);
  clscrs_splist_append(attrlist, (const oratext *)clscrs_TYPE_NAME, NULL);
  clscrs_splist_append(attrlist, (const oratext *)clscrs_BASE_TYPE, NULL);
  status = clscrs_stat2(NULL, clscrs_entity_restype, NULL, NULL, 
               attrlist, 0,
               out_entitylist);
  print_entitylist(out_entitylist, TRUE);

  clscrs_splist_destroy(&attrlist);
  clscrs_reslist_destroy(&out_entitylist);

  /* Create a simple filter BASE_TYPE == clscrs.resourcetype2.demo */
  clscrs_compfilter_create(ctx, clscrs_BASE_TYPE, 
                           comp_eq, RESOURCE_TYPE_2, &filter1);
  clscrs_filter_print(filter1);
  
  /* Stat resource types matching the filter specified in filter1 */
  clscrs_reslist_create(ctx, &out_entitylist);
  status = clscrs_stat2(NULL, clscrs_entity_restype, filter1, NULL, 
                        NULL, 0,
                        out_entitylist);
  print_entitylist(out_entitylist, TRUE);

  clscrs_reslist_destroy(&out_entitylist);

  clscrs_filter_destroy(&filter1);

}

/*
  This function is used to get status of resources. It employs both filters 
  as well as regular resource names.

  APIs Used:
    clscrs_stat2
    clscrs_compfilter_create
    clscrs_exprfilter_create
    clscrs_filter_print
*/
void stat_resources(clscrs_ctx *ctx)
{
  clscrs_reslist   *out_entitylist;
  clscrs_splist    *attrlist, *rqlist;
  CLSCRS_STAT       status;
  clscrs_filter    *filter1     = NULL, *filter2 = NULL;
  clscrs_filter    *expr_filter = NULL;
  clscrs_comparator comp_eq     = clscrs_comparator_eq;
  clscrs_operator   and_filter  = clscrs_operator_and;

  printf("\nQuerying resources\n");

  /* Stat a single resource asking for all attributes*/
  clscrs_reslist_create(ctx, &out_entitylist);
  clscrs_splist_create(ctx, &rqlist);
  clscrs_splist_append(rqlist, RESOURCE_1, NULL);
  status = clscrs_stat2(rqlist, clscrs_entity_res, NULL, NULL, 
               NULL, 0,
               out_entitylist);
  print_entitylist(out_entitylist, FALSE);

  clscrs_splist_destroy(&rqlist);
  clscrs_reslist_destroy(&out_entitylist);

  /* Stat all resources asking for specific attributes*/
  clscrs_reslist_create(ctx, &out_entitylist);
  clscrs_splist_create(ctx, &attrlist);
  clscrs_splist_append(attrlist, (const oratext *)clscrs_NAME, NULL);
  clscrs_splist_append(attrlist, (const oratext *)clscrs_TYPE, NULL);
  clscrs_splist_append(attrlist, (const oratext *)clscrs_CHECK_INTERVAL, NULL);
  status = clscrs_stat2(NULL, clscrs_entity_res, NULL, NULL, 
               attrlist, 0,
               out_entitylist);
  print_entitylist(out_entitylist, FALSE);

  clscrs_splist_destroy(&attrlist);
  clscrs_reslist_destroy(&out_entitylist);

  /* Create a simple filter TYPE == clscrs.resourcetype2.demo */
  clscrs_compfilter_create(ctx, clscrs_TYPE, 
                           comp_eq, RESOURCE_TYPE_2, &filter1);
  clscrs_filter_print(filter1);
  
  /* Stat resources matching the filter specified in filter1 */
  clscrs_reslist_create(ctx, &out_entitylist);
  status = clscrs_stat2(NULL, clscrs_entity_res, filter1, NULL, 
               NULL, 0,
               out_entitylist);
  print_entitylist(out_entitylist, FALSE);

  clscrs_reslist_destroy(&out_entitylist);

  /* Create another simple filter PLACEMENT == balanced */
  clscrs_compfilter_create(ctx, clscrs_PLACEMENT, 
                              comp_eq, "balanced", &filter2);
  clscrs_filter_print(filter2);
  
  /* Create an expression filter:
     TYPE == clscrs.resourcetype1.demo && PLACEMENT == balanced 
  */
  clscrs_exprfilter_create(filter1, and_filter, filter2, &expr_filter);
  clscrs_filter_print(expr_filter);

  /* Stat resources matching the expression filter specified in expr_filter */
  clscrs_reslist_create(ctx, &out_entitylist);
  status = clscrs_stat2(NULL, clscrs_entity_res, expr_filter, NULL, 
               NULL, 0,
               out_entitylist);
  print_entitylist(out_entitylist, FALSE);

  clscrs_reslist_destroy(&out_entitylist);

  clscrs_filter_destroy(&filter1);
  clscrs_filter_destroy(&filter2);
  clscrs_filter_destroy(&expr_filter);

  /* Stat a single instance of a resource having multiple cardinality */
  /* Create a simple filter NAME == clscrs.resource3.demo */
  clscrs_compfilter_create(ctx, clscrs_NAME, 
                           comp_eq, RESOURCE_3, &filter1);
  clscrs_filter_print(filter1);

  /* Create another simple filter CARDINALITY_ID == 1 */
  clscrs_compfilter_create(ctx, clscrs_CARDINALITY_ID, 
                              comp_eq, "1", &filter2);
  clscrs_filter_print(filter2);
  
  /* Create an expression filter:
     NAME == clscrs.resource3.demo && CARDINALITY_ID == 1 
  */
  clscrs_exprfilter_create(filter1, and_filter, filter2, &expr_filter);
  clscrs_filter_print(expr_filter);

  /* Stat a single resource instance */
  clscrs_reslist_create(ctx, &out_entitylist);
  status = clscrs_stat2(NULL, clscrs_entity_res, expr_filter, NULL, 
               NULL, 0,
               out_entitylist);
  print_entitylist(out_entitylist, FALSE);

  clscrs_reslist_destroy(&out_entitylist);

  clscrs_filter_destroy(&filter1);
  clscrs_filter_destroy(&filter2);
  clscrs_filter_destroy(&expr_filter);
}

/*
  This function is used to get status of server pools.

  APIs Used:
    clscrs_stat2
*/
void stat_server_pools(clscrs_ctx *ctx)
{
  clscrs_reslist   *out_entitylist;
  clscrs_splist    *attrlist, *rqlist;
  CLSCRS_STAT       status;
  oratext          *sp_id = NULL;

  printf("\nQuerying server pools \n");

  /* Stat a single server pool asking for all attributes*/
  clscrs_reslist_create(ctx, &out_entitylist);
  clscrs_entity_id_create(ctx, SERVER_POOL_1, clscrs_entity_serverpool, &sp_id);
  clscrs_splist_create(ctx, &rqlist);
  clscrs_splist_append(rqlist, sp_id, NULL);
  status = clscrs_stat2(rqlist, clscrs_entity_serverpool, NULL, NULL, 
               NULL, 0,
               out_entitylist);
  print_entitylist(out_entitylist, FALSE);

  clscrs_splist_destroy(&rqlist);
  clscrs_reslist_destroy(&out_entitylist);
  clscrs_entity_id_destroy(ctx, &sp_id);

  /* Stat all server pools asking for specific attributes*/
  clscrs_reslist_create(ctx, &out_entitylist);
  clscrs_splist_create(ctx, &attrlist);
  clscrs_splist_append(attrlist, (const oratext *)clscrs_NAME, NULL);
  clscrs_splist_append(attrlist, (const oratext *)clscrs_ACTIVE_SERVERS, NULL);
  status = clscrs_stat2(NULL, clscrs_entity_serverpool, NULL, NULL, 
               attrlist, 0,
               out_entitylist);
  print_entitylist(out_entitylist, FALSE);

  clscrs_splist_destroy(&attrlist);
  clscrs_reslist_destroy(&out_entitylist);

}

/*
  This function is used to get status of servers.

  APIs Used:
    clscrs_stat2
*/
void stat_servers(clscrs_ctx *ctx)
{
  clscrs_reslist   *out_entitylist;
  clscrs_splist    *attrlist, *rqlist;
  CLSCRS_STAT       status;

  printf("\nQuerying servers \n");

  /* Stat all server pools asking for specific attributes*/
  clscrs_reslist_create(ctx, &out_entitylist);
  clscrs_splist_create(ctx, &attrlist);
  clscrs_splist_append(attrlist, (const oratext *)clscrs_NAME, NULL);
  clscrs_splist_append(attrlist, (const oratext *)clscrs_ACTIVE_POOLS, NULL);
  clscrs_splist_append(attrlist, (const oratext *)clscrs_SERVER_STATE, NULL);
  status = clscrs_stat2(NULL, clscrs_entity_server, NULL, NULL, 
               attrlist, 0,
               out_entitylist);
  print_entitylist(out_entitylist, FALSE);

  clscrs_splist_destroy(&attrlist);
  clscrs_reslist_destroy(&out_entitylist);

}

/* This function is used to ask the Oracle Clusterware to run a check
   action on the specified resource.

   APIs Used:
    clscrs_check_resource2
    clscrs_entity_id_create
    clscrs_reslist_create
    clscrs_splist_create
    clscrs_splist_append
    clscrs_reslist_destroy
    clscrs_splist_destroy
*/
CLSCRS_STAT check_resources(clscrs_ctx *ctx)
{
  oratext        *res_id1    = NULL, *res_id2 = NULL;
  clscrs_splist  *resid_list = NULL;
  clscrs_reslist *op_status  = NULL;
  CLSCRS_STAT     status;

  printf("Executing check action for resources\n");

  clscrs_entity_id_create(ctx, RESOURCE_1, clscrs_entity_res, &res_id1);
  clscrs_entity_id_create(ctx, RESOURCE_2, clscrs_entity_res, &res_id2);

  clscrs_splist_create(ctx, &resid_list);
  clscrs_splist_append(resid_list, res_id1, NULL);
  clscrs_splist_append(resid_list, res_id2, NULL);

  clscrs_reslist_create(ctx, &op_status);

  status = clscrs_check_resource2(resid_list, NULL, 0,
                                  (clscrs_msgf2)client_callback,
                                  0, op_status);

  printf("\n Check action status = %d\n", status);
  
  print_opstatus(op_status);

  clscrs_reslist_destroy(&op_status);
  clscrs_splist_destroy(&resid_list);
  clscrs_entity_id_destroy(ctx, &res_id1);
  clscrs_entity_id_destroy(ctx, &res_id2);

  return status;
}

/*
 This function modifies the attributes of a demo resource already 
 registered with Oracle Clusterware.
 NOTE: Similar idea can be applied to modify resource type and server pool
 attributes.

 APIs Used:
  clscrs_register_resource2
  clscrs_reslist_create
  clscrs_reslist_destroy
*/

CLSCRS_STAT modify_resource(clscrs_ctx *ctx) 
{
  clscrs_reslist *reslist = NULL, *op_status = NULL;
  clscrs_splist  *rqlist = NULL;
  CLSCRS_STAT     status;
  clscrs_res     *res;

  printf("Modifying a resource\n");

  clscrs_res_create(ctx, RESOURCE_5, &res);

  clscrs_res_set_attr(res, clscrs_CHECK_INTERVAL, (const oratext*)"600");

  clscrs_reslist_create(ctx, &reslist);

  clscrs_reslist_append(reslist, res);

  clscrs_reslist_create(ctx, &op_status);

  status = clscrs_register_resource2(reslist, CLSCRS_FLAG_REG_UPDATE,
                                     (clscrs_msgf2)client_callback,
                                     NULL, op_status);

  printf("\nModify resource status = %d\n", status);
  
  print_opstatus(op_status);
  
  clscrs_reslist_destroy(&op_status);
  clscrs_reslist_destroy(&reslist);
  
  /* Stat the modified resource */
  clscrs_reslist_create(ctx, &op_status);
  clscrs_splist_create(ctx, &rqlist);
  clscrs_splist_append(rqlist, RESOURCE_5, NULL);
  status = clscrs_stat2(rqlist, clscrs_entity_res, NULL, NULL, 
                        NULL, 0,
                        op_status);
  print_entitylist(op_status, FALSE);

  clscrs_splist_destroy(&rqlist);
  clscrs_reslist_destroy(&op_status);

  return status;
}

/*
  This function is used to relocate a resource registered with Oracle 
  Clusterware. 

  APIs Used:
    clscrs_relocate_resource2
    clscrs_env_create
    clscrs_entity_id_create
*/
CLSCRS_STAT relocate_resource(clscrs_ctx *ctx)
{
  oratext          *res_id1     = NULL;
  clscrs_reslist   *op_status   = NULL;
  clscrs_splist    *resid_list  = NULL;
  CLSCRS_STAT       status;
  clscrs_env        env;
  oratext           server_name[20] = "dummy server";

  printf("\nRelocating a resource\n");

  clscrs_env_create(ctx, &env);

  /* Relocate a resource */
  clscrs_entity_id_create(ctx, RESOURCE_1, clscrs_entity_res, &res_id1);

  clscrs_splist_create(ctx, &resid_list);
  clscrs_splist_append(resid_list, res_id1, NULL);

  clscrs_reslist_create(ctx, &op_status);

  status = clscrs_relocate_resource2(resid_list, server_name, env,
                                     (clscrs_msgf2)client_callback, 
                                     NULL, 0,
                                     op_status);

  print_opstatus(op_status);

  clscrs_splist_destroy(&resid_list);
  clscrs_reslist_destroy(&op_status);
  clscrs_entity_id_destroy(ctx, &res_id1);
  clscrs_env_delete(env);

  return status;
}

/*
  This function is used to stop the resources registered with Oracle 
  Clusterware. It employs both filters as well as regular resource names.

  APIs Used:
    clscrs_stop_resource2
    clscrs_compfilter_create
    clscrs_exprfilter_create
    clscrs_filter_print
    clscrs_filter_destroy
    clscrs_env_create
    clscrs_entity_id_create
*/
CLSCRS_STAT stop_resources(clscrs_ctx *ctx)
{
  oratext          *res_id1     = NULL;
  clscrs_reslist   *op_status   = NULL;
  clscrs_splist    *resid_list  = NULL;
  CLSCRS_STAT       status;
  clscrs_filter    *filter1     = NULL, *filter2 = NULL;
  clscrs_filter    *expr_filter = NULL;
  clscrs_comparator comp_ne     = clscrs_comparator_ne;
  clscrs_comparator comp_eq     = clscrs_comparator_eq;
  clscrs_operator   and_filter  = clscrs_operator_and;
  clscrs_env        env;

  printf("\nStopping resources\n");

  clscrs_env_create(ctx, &env);

  /* Stop a single resource */
  clscrs_entity_id_create(ctx, RESOURCE_1, clscrs_entity_res, &res_id1);

  clscrs_splist_create(ctx, &resid_list);
  clscrs_splist_append(resid_list, res_id1, NULL);

  clscrs_reslist_create(ctx, &op_status);

  status = clscrs_stop_resource2(resid_list, env, 
                                 (clscrs_msgf2)client_callback, 
                                 NULL, 0,
                                 op_status);

  print_opstatus(op_status);

  clscrs_reslist_destroy(&op_status);
  clscrs_splist_destroy(&resid_list);
  clscrs_entity_id_destroy(ctx, &res_id1);

  /* Create a simple filter TYPE == clscrs.resourcetype1.demo */
  clscrs_compfilter_create(ctx, clscrs_TYPE, 
                           comp_eq, RESOURCE_TYPE_2, &filter1);
  clscrs_filter_print(filter1);
  
  /* Create another simple filter PLACEMENT != balanced */
  clscrs_compfilter_create(ctx, clscrs_PLACEMENT, 
                           comp_ne, "balanced", &filter2);
  clscrs_filter_print(filter2);
  
  /* Create an expression filter:
     TYPE == clscrs.resourcetype1.demo && PLACEMENT != balanced 
  */
  clscrs_exprfilter_create(filter1, and_filter, filter2, &expr_filter);
  clscrs_filter_print(expr_filter);

  /* Stop resources matching the expression filter specified in expr_filter */
  clscrs_reslist_create(ctx, &op_status);
  status = clscrs_stop_resource2(expr_filter, env, 
                                 (clscrs_msgf2)client_callback, 
                                 NULL, 0, op_status);
  print_opstatus(op_status);

  clscrs_filter_destroy(&filter1);
  clscrs_filter_destroy(&filter2);
  clscrs_filter_destroy(&expr_filter);
  clscrs_reslist_destroy(&op_status);

  /* Stop a single instance of a resource having multiple cardinality */
  /* Create a simple filter NAME == clscrs.resource3.demo */
  clscrs_compfilter_create(ctx, clscrs_NAME, 
                           comp_eq, RESOURCE_3, &filter1);
  clscrs_filter_print(filter1);
  
  /* Create another simple filter CARDINALITY_ID == 1 */
  clscrs_compfilter_create(ctx, clscrs_CARDINALITY_ID, 
                              comp_eq, "1", &filter2);
  clscrs_filter_print(filter2);
  
  /* Create an expression filter:
     NAME == clscrs.resource3.demo && CARDINALITY_ID == 1 
  */
  clscrs_exprfilter_create(filter1, and_filter, filter2, &expr_filter);
  clscrs_filter_print(expr_filter);

  /* Stop a single instance of a resource */
  clscrs_reslist_create(ctx, &op_status);
  status = clscrs_stop_resource2(expr_filter, env, 
                                 (clscrs_msgf2)client_callback, 
                                 NULL, 0, op_status);
  print_opstatus(op_status);

  clscrs_filter_destroy(&filter1);
  clscrs_filter_destroy(&filter2);
  clscrs_filter_destroy(&expr_filter);
  clscrs_reslist_destroy(&op_status);

  /* Stop all resources */
  clscrs_splist_create(ctx, &resid_list);
  clscrs_reslist_create(ctx, &op_status);

  status = clscrs_stop_resource2(resid_list, env, 
                                 (clscrs_msgf2)client_callback, 
                                 NULL, CLSCRS_FLAG_ALL,
                                 op_status);

  print_opstatus(op_status);
  clscrs_reslist_destroy(&op_status);
  clscrs_splist_destroy(&resid_list);
  clscrs_env_delete(env);

  return status;
}

/*
  This function is used to unregister resources that have been registered with 
  Oracle clusterware.

  APIs Used:
    clscrs_unregister_resource2
    clscrs_reslist_create
    clscrs_splist_create
*/
CLSCRS_STAT unregister_resources(clscrs_ctx *ctx)
{
  clscrs_splist  *resname_list = NULL;
  clscrs_reslist *op_status    = NULL;
  CLSCRS_STAT     status;

  printf("Unregistering resources\n");

  clscrs_splist_create(ctx, &resname_list);
  clscrs_splist_append(resname_list, RESOURCE_1, NULL);
  clscrs_splist_append(resname_list, RESOURCE_2, NULL);
  clscrs_splist_append(resname_list, RESOURCE_3, NULL);
  clscrs_splist_append(resname_list, RESOURCE_4, NULL);
  clscrs_splist_append(resname_list, RESOURCE_5, NULL);

  clscrs_reslist_create(ctx, &op_status);

  status = clscrs_unregister_resource2(resname_list, 0,
                                       (clscrs_msgf2)client_callback,
                                       NULL, op_status);

  printf("\nUnregister resources status = %d\n", status);
  
  print_opstatus(op_status);

  clscrs_reslist_destroy(&op_status);
  clscrs_splist_destroy(&resname_list);

  return status;
}

/*
  This function is used to unregister resource types that have been registered
  with Oracle clusterware.

  APIs Used:
    clscrs_unregister_type
    clscrs_reslist_create
    clscrs_splist_create
*/
CLSCRS_STAT unregister_resource_types(clscrs_ctx *ctx)
{
  clscrs_splist  *restype_list = NULL;
  clscrs_reslist *op_status    = NULL;
  CLSCRS_STAT     status;

  printf("Unregistering resource types\n");

  clscrs_splist_create(ctx, &restype_list);
  clscrs_splist_append(restype_list, RESOURCE_TYPE_1, NULL);
  clscrs_splist_append(restype_list, RESOURCE_TYPE_2, NULL);

  clscrs_reslist_create(ctx, &op_status);

  status = clscrs_unregister_type(restype_list, 0,
                                  (clscrs_msgf2)client_callback,
                                  NULL, op_status);

  printf("\nUnregister resource type status = %d\n", status);
  
  print_opstatus(op_status);

  clscrs_reslist_destroy(&op_status);
  clscrs_splist_destroy(&restype_list);

  return status;
}

/*
  This function is used to unregister server pools that have been registered
  with Oracle clusterware.

  APIs Used:
    clscrs_unregister_serverpool
    clscrs_reslist_create
    clscrs_splist_create
*/
CLSCRS_STAT unregister_server_pools(clscrs_ctx *ctx)
{
  clscrs_splist  *sp_list   = NULL;
  clscrs_reslist *op_status = NULL;
  CLSCRS_STAT     status;

  printf("Unregistering server pool\n");

  clscrs_splist_create(ctx, &sp_list);
  clscrs_splist_append(sp_list, SERVER_POOL_1, NULL);

  clscrs_reslist_create(ctx, &op_status);

  status = clscrs_unregister_serverpool(sp_list, 0,
                                        (clscrs_msgf2)client_callback,
                                        NULL, op_status);

  printf("\nUnregister server pool status = %d\n", status);
  
  print_opstatus(op_status);

  clscrs_reslist_destroy(&op_status);
  clscrs_splist_destroy(&sp_list);

  return status;
}

void stat_3(clscrs_ctx *ctx)
{
  clscrs_crsentitylist   *out_entitylist;
  clscrs_splist    *attrlist, *rqlist;
  CLSCRS_STAT       status;
  clscrs_filter    *filter1     = NULL, *filter2 = NULL, *filter3 = NULL;
  clscrs_comparator comp_eq     = clscrs_comparator_eq;
  clscrs_query     *q1, *q2, *q3;
  clscrs_querylist *querylist;

  printf("\n**********************************Stat 3 starts *******************************\n");

  clscrs_crsentitylist_create(ctx, &out_entitylist, (const oratext*)NULL);

  clscrs_querylist_create(ctx, &querylist);

  /* First Query - for Resources */
  clscrs_query_create(ctx, clscrs_entity_res, &q1);

  clscrs_query_set_attr(q1, (const oratext *)clscrs_NAME, NULL);
  clscrs_query_set_attr(q1, (const oratext *)clscrs_TYPE, NULL);
  clscrs_query_set_attr(q1, (const oratext *)clscrs_CHECK_INTERVAL, NULL);

  /* Create a simple filter TYPE == clscrs.resourcetype2.demo */
  clscrs_compfilter_create(ctx, clscrs_TYPE, 
                           comp_eq, RESOURCE_TYPE_2, &filter1);

  clscrs_query_set_filter(q1, filter1);

  clscrs_querylist_append(querylist, q1);
  
  /* Second Query - for Resource Types */
  clscrs_query_create(ctx, clscrs_entity_restype, &q2);

  clscrs_query_set_attr(q2, (const oratext*)clscrs_AUTO_START, NULL);
  clscrs_query_set_attr(q2, (const oratext*)clscrs_BASE_TYPE, NULL);

  /* Create a simple filter BASE_TYPE == clscrs.resourcetype2.demo */
  clscrs_compfilter_create(ctx, clscrs_BASE_TYPE, 
                           comp_eq, "cluster_resource", &filter2);
  
  clscrs_query_set_filter(q2, filter2);

  clscrs_querylist_append(querylist, q2);

  /* Third Query - for Server Pools */
  clscrs_query_create(ctx, clscrs_entity_serverpool, &q3);

  clscrs_query_set_attr(q3, (const oratext*)clscrs_NAME, NULL);
  clscrs_query_set_attr(q3, (const oratext*)clscrs_ACTIVE_SERVERS, NULL);

  /* Create a simple filter NAME == clscrs.serverpool1.demo */
  clscrs_compfilter_create(ctx, clscrs_NAME, 
                           comp_eq, SERVER_POOL_1, &filter3);
  
  clscrs_query_set_filter(q3, filter3);

  clscrs_querylist_append(querylist, q3);

  /* Stat 3 API call */
  status = clscrs_stat3(querylist, 0, out_entitylist);

  printf("\nStat3 status = %d\n", status);

  /* Process and print output */
  print_entitylist_stat3(out_entitylist);

  /* Cleanup */
  clscrs_filter_destroy(&filter1);
  clscrs_filter_destroy(&filter2);
  clscrs_filter_destroy(&filter3);
  clscrs_querylist_destroy(&querylist);
  clscrs_crsentitylist_destroy(&out_entitylist);

  printf("\n**********************************Stat 3 ends *******************************\n");

}

void crsapp_print(void *usrp, const oratext * msg, sword msglen)
{
  print(usrp, (oratext *)"%s", msg);
}

/* This function orchestrates the whole demo application.  */

int main(int argc, char *argv[])
{
  clscrs_reslist *res_list         = NULL, *res_type_list = NULL;
  clscrs_reslist *server_pool_list = NULL;
  uword           flags            = 0;
  clscrs_ctx     *ctx              = NULL;
  clscrs_msgf     tracefunc        = crsapp_print;
  void            *tracectx        = NULL;
  ub4             count;
  CLSCRS_STAT     status;

  if (argc > 1 && !strcmp(argv[1],"trace"))
  {
    flags = CLSCRS_FLAG_TRACE;
  }

  /* Initialize the clscrs context */
  if (CLSCRS_STAT_SUCCESS != 
      (status = clscrs_init_crs(&ctx, tracefunc, tracectx, flags)))
  {
    printf("clscrs_init_crs failed with status %d\n", status);
    exit(1);
  }

  /* Create the input resource list */
  clscrs_reslist_create(ctx, &res_list);

  /* Create the input resource type list */
  clscrs_reslist_create(ctx, &res_type_list);

  /* Create the input server pool list */
  clscrs_reslist_create(ctx, &server_pool_list);

  /* Create the demo CRS resources, resource types and server pool */
  create_crs_entities(ctx, res_list, res_type_list, server_pool_list); 

  /* Register Resource Types */
  register_resource_types(ctx, res_type_list);
  printf(SEPARATOR);

  /* Register Resources */
  register_resources(ctx, res_list);
  printf(SEPARATOR);

  /* Register Server Pools */
  register_server_pools(ctx, server_pool_list);
  printf(SEPARATOR);

  /* Start the Resources */
  start_resources(ctx);
  printf(SEPARATOR);

  /* Stat the Resource Types */
  stat_resource_types(ctx);
  printf(SEPARATOR);

  /* Stat the Resources */
  stat_resources(ctx);
  printf(SEPARATOR);

  /* Stat the Server Pools */
  stat_server_pools(ctx);
  printf(SEPARATOR);

  /* Stat the Servers */
  stat_servers(ctx);
  printf(SEPARATOR);

  /* Check the resources */
  check_resources(ctx);
  printf(SEPARATOR);

  /* Modify a resource */
  modify_resource(ctx);
  printf(SEPARATOR);

  /* Relocate a Resource */
  relocate_resource(ctx);
  printf(SEPARATOR);

  /* Stop the resources */
  stop_resources(ctx);
  printf(SEPARATOR);

  /* Stat the CRS Entities */
  stat_3(ctx);
  printf(SEPARATOR);

  /* Unregister the resources */
  unregister_resources(ctx);
  printf(SEPARATOR);
  
  /* Unregister the resource types */
  unregister_resource_types(ctx);
  printf(SEPARATOR);
  
  /* Unregister the server pools */
  unregister_server_pools(ctx);
  printf(SEPARATOR);
  
  /* Destroy the input resource list */
  clscrs_reslist_destroy(&res_list);

  /* Destroy the input resource type list */
  clscrs_reslist_destroy(&res_type_list);

  /* Destroy the input server pool list */
  clscrs_reslist_destroy(&server_pool_list);

  /* Terminate the clscrs context */
  clscrs_term_crs(&ctx);

  printf("\n");
  return 0;
}

/* end of file crsapp.c */

